<?php
/**
 * ProductImageListDeleteAjaxController.inc.php 2020-2-4
 * Last Modified: 2/4/20, 10:41 AM
 * Gambio GmbH
 * http://www.gambio.de
 * Copyright (c) 2020 Gambio GmbH
 * Released under the GNU General Public License (Version 2)
 * [http://www.gnu.org/licenses/gpl-2.0.html]
 */

declare(strict_types=1);

use Gambio\ProductImageList\Image\ValueObjects\Id;
use Gambio\ProductImageList\ImageList\ValueObjects\ListModifierId;
use Gambio\ProductImageList\ImageList\ValueObjects\ListModifierType;
use Gambio\ProductImageList\ImageList\ValueObjects\ListId;
use Gambio\ProductImageList\DeleteService\Exceptions\ListIsUsedForACombinationException;
use Gambio\ProductImageList\DeleteService\Exceptions\ListIsUsedForAnAttributeException;
use Gambio\ProductImageList\Interfaces\ProductImageListDeleteServiceInterface;

/**
 * Class ProductImageListDeleteAjaxController
 * Available Endpoints:
 * [DELETE] admin/admin.php?do=ProductImageListDeleteAjax/deleteImageListById&id=<ImageListId>
 * [DELETE] admin/admin.php?do=ProductImageListDeleteAjax/deleteImageById&id=<Comma,Separated,Ids>
 */
class ProductImageListDeleteAjaxController extends AdminHttpViewController
{
    /**
     * @var ProductImageListDeleteServiceInterface
     */
    protected $productImageListDeleteService;
    
    /**
     * @var LanguageTextManager
     */
    protected $languageTextManager;
    
    
    /**
     * Init
     */
    public function init()
    {
        $this->productImageListDeleteService = StaticGXCoreLoader::getService('ProductImageListDelete');
        $this->languageTextManager           = MainFactory::create('LanguageTextManager', 'product_image_lists');
    }
    
    
    public function actionDeleteImageListById() : JsonHttpControllerResponse
    {
        $success = false;
    
        $id           = (int)$this->_getQueryParameter('id');
        $modifierId   = (int)$this->_getQueryParameter('modifierId');
        $modifierType = (string)$this->_getQueryParameter('modifierType');
        try {
            if (!$this->isValidRequestMethod('delete')) {
                throw new Exception(
                    $this->getTranslatedText('CONTROLLER_MESSAGE_INVALID_REQUEST_METHOD')
                );
            }
            if ($id === 0 || $modifierId === 0 || !$modifierType) {
                throw new InvalidArgumentException(
                    $this->getTranslatedText('CONTROLLER_MESSAGE_MISSING_JSON_REQUIRED_INPUT')
                );
            }
            $listId           = new ListId($id);
            $listModifierId   = new ListModifierId($modifierId);
            $listModifierType = new ListModifierType($modifierType);
            $this->productImageListDeleteService->deleteImageList($listId, $listModifierId, $listModifierType);
            $success = true;
            $message = $this->getTranslatedText('CONTROLLER_MESSAGE_LIST_DELETED');
        } catch (ListIsUsedForACombinationException $e) {
            $message = $this->getTranslatedText('CONTROLLER_MESSAGE_LIST_USED_BY_COMBI');
        } catch (ListIsUsedForAnAttributeException $e) {
            $message = $this->getTranslatedText('CONTROLLER_MESSAGE_LIST_USED_BY_ATTRIBUTE');
        } catch (Throwable $e) {
            $message = $e->getMessage();
        }
    
        $response = [
            'success' => $success,
            'message' => $message,
        ];
    
        return new JsonHttpControllerResponse($response);
    }
    
    
    public function actionDeleteImageById() : JsonHttpControllerResponse
    {
        $success = false;
        try {
            if (!$this->isValidRequestMethod('delete')) {
                throw new Exception(
                    $this->getTranslatedText('CONTROLLER_MESSAGE_INVALID_REQUEST_METHOD')
                );
            }
            $ids = $this->_getIdObjectFromQueryParameters();
            if (!$ids) {
                throw new InvalidArgumentException(
                    $this->getTranslatedText('CONTROLLER_MESSAGE_MISSING_JSON_REQUIRED_INPUT') . ' "Image IDs"'
                );
            }
            $this->productImageListDeleteService->deleteImage(...$ids);
            $success = true;
            $message = $this->getTranslatedText('CONTROLLER_MESSAGE_IMAGES_DELETED');
        } catch (Throwable $e) {
            $message = $e->getMessage();
        }
        $response = [
            'success' => $success,
            'message' => $message,
        ];
        
        return new JsonHttpControllerResponse($response);
    }
    
    
    protected function _getIdObjectFromQueryParameters()
    {
        $param    = $this->_getQueryParameter('id');
        $queryIds = $param ? explode(',', $param) : [];
        $ids      = [];
        foreach ($queryIds as $id) {
            $ids[] = new Id((int)$id);
        }
        
        return $ids;
    }
    
    
    /**
     * @param string $phraseName
     *
     * @return string
     */
    private function getTranslatedText(string $phraseName) : string
    {
        return $this->languageTextManager->get_text($phraseName);
    }
    
}